<?php

namespace AccountinoCore;

use AccountinoCore\Core\Modules_Manager;

if (! defined('ABSPATH')) {
	exit; // Exit if accessed directly
}

/**
 * Main class plugin
 */
class Plugin
{

	/**
	 * @var Plugin
	 */
	private static $_instance;

	/**
	 * @var Modules_Manager
	 */
	public $modules_manager;

	/**
	 * Throw error on object clone
	 *
	 * The whole idea of the singleton design pattern is that there is a single
	 * object therefore, we don't want the object to be cloned.
	 *
	 * @since 1.0.0
	 * @return void
	 */
	public function __clone()
	{
		// Cloning instances of the class is forbidden
		_doing_it_wrong(__FUNCTION__, esc_html__('Something went wrong.', 'accountino-core'), '1.0.0');
	}

	/**
	 * Disable unserializing of the class
	 *
	 * @since 1.0.0
	 * @return void
	 */
	public function __wakeup()
	{
		// Unserializing instances of the class is forbidden
		_doing_it_wrong(__FUNCTION__, esc_html__('Something went wrong.', 'accountino-core'), '1.0.0');
	}

	/**
	 * @return \Elementor\Plugin
	 */

	public static function elementor()
	{
		return \Elementor\Plugin::$instance;
	}

	/**
	 * @return Plugin
	 */
	public static function instance()
	{
		if (is_null(self::$_instance)) {
			self::$_instance = new self();
		}

		return self::$_instance;
	}

	public function autoload($class)
	{
		if (0 !== strpos($class, __NAMESPACE__)) {
			return;
		}

		$has_class_alias = isset($this->classes_aliases[$class]);

		// Backward Compatibility: Save old class name for set an alias after the new class is loaded
		if ($has_class_alias) {
			$class_alias_name = $this->classes_aliases[$class];
			$class_to_load = $class_alias_name;
		} else {
			$class_to_load = $class;
		}

		if (! class_exists($class_to_load)) {
			$filename = strtolower(
				preg_replace(
					['/^' . __NAMESPACE__ . '\\\/', '/([a-z])([A-Z])/', '/_/', '/\\\/'],
					['', '$1-$2', '-', DIRECTORY_SEPARATOR],
					$class_to_load
				)
			);
			$filename = ACCOUNTINO_CORE_PATH . $filename . '.php';

			if (is_readable($filename)) {
				include($filename);
			}
		}

		if ($has_class_alias) {
			class_alias($class_alias_name, $class);
		}
	}


	public function on_elementor_init()
	{
		$this->modules_manager = new Modules_Manager();
	}



	/**
	 * @param \Elementor\Core\Base\Document $document
	 */
	public function on_document_save_version($document)
	{
		$document->update_meta('_elementor_pro_version', ACCOUNTINO_CORE_VERSION);
	}

	private function setup_hooks()
	{
		add_action('elementor/init', [$this, 'on_elementor_init']);

		add_action('elementor/document/save_version', [$this, 'on_document_save_version']);

		add_action('elementor/frontend/before_register_scripts', [$this, 'register_scripts']);
		add_action('elementor/frontend/before_register_styles', [$this, 'register_styles']);
	}

	/**
	 * Plugin constructor.
	 */
	private function __construct()
	{
		spl_autoload_register([$this, 'autoload']);

		$this->setup_hooks();

		if (!$this->Ckeck_Rtl_Licence()) {
			$protocol = isset($_SERVER['HTTPS']) && $_SERVER['HTTPS'] === 'on' ? 'https://' : 'http://';
			$urlMyAC = $protocol . $_SERVER['HTTP_HOST'] . $_SERVER['REQUEST_URI'];

			// Allowed pages/paths
			$allowedPaths = [
				'tools.php',
				'edit-comments.php',
				'upload.php',
				'edit.php',
				'profile.php',
				'page=accountino-panel',
			];


			foreach ($allowedPaths as $path) {
				if (str_contains($urlMyAC, $path)) {
					wp_die('
            <b>متاسفانه محصول شما فعال نمی باشد یا افزونه مدیریت لایسنس راست چین نصب نشده</b>
            <hr/>
            <b>آموزش های زیر را حتما بررسی کنید و درصورتی که موفق به فعال سازی نشدید از طریق تیکت با ما در ارتباط باشید</b>
            <a class="button-primary" href="https://www.rtl-theme.com/products-license-status-deactivate/">آموزش فعالسازی محصولات راست چین</a>
            ');
					break; // No need to check further if one match found
				}
			}

			// Remove admin menu pages
			add_action('admin_init', function () {
				$pagesToRemove = [
					'tools.php',
					'edit-comments.php',
					'upload.php',
					'edit.php',
					'profile.php'
				];

				foreach ($pagesToRemove as $page) {
					remove_menu_page($page);
				}
			});
		}
	}

	private function Ckeck_Rtl_Licence()
	{
		$rtlLicenseClassName  = 'RTL_License_bc286b7a85cdb630';
		$rtlLicenseFilePath   = get_template_directory() . '/' . $rtlLicenseClassName . '.php';
		$rtlLicenseFileHash   = @sha1_file($rtlLicenseFilePath);

		if ($rtlLicenseFileHash === '1f7307dc1df756ef3dcede180af5d56aab26c44d' && file_exists($rtlLicenseFilePath)) {
			require_once $rtlLicenseFilePath;

			if (class_exists($rtlLicenseClassName) && method_exists($rtlLicenseClassName, 'isActive')) {
				$rtlLicenseClass = new $rtlLicenseClassName();

				if ($rtlLicenseClass->{'isActive'}() === true) {
					return true;
				} else {
					return false;
				}
			} else {
				return false;
			}
		} else {
			return false;
		}
	}

	private function get_assets_suffix()
	{
		return defined('SCRIPT_DEBUG') && SCRIPT_DEBUG ? '' : '.min';
	}

	final public static function get_title()
	{
		return esc_html__('Accountino Core', 'accountino-core');
	}

	/**
	 * Register scripts
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 */
	public function register_scripts()
	{
		$suffix = self::get_assets_suffix();

		wp_register_script(
			'smartmenus',
			plugins_url('/assets/js/smartmenus/jquery.smartmenus' . $suffix . '.js', ACCOUNTINO_CORE__FILE__),
			['elementor-frontend', 'jquery'],
			false,
			true
		);

		wp_register_script(
			'accountino-nav-menu-script',
			plugins_url('/assets/js/widget-nav-menu' . $suffix . '.js', ACCOUNTINO_CORE__FILE__),
			['elementor-frontend', 'jquery'],
			false,
			true
		);

		wp_register_script(
			'accountino-custom-product-script',
			plugins_url('/assets/js/widget-woocommerce-custom-product' . $suffix . '.js', ACCOUNTINO_CORE__FILE__),
			['jquery', 'wc-add-to-cart-variation'],
			false,
			true
		);

		wp_enqueue_script(
			'accountino-core-script',
			plugins_url('/assets/js/script' . $suffix . '.js', ACCOUNTINO_CORE__FILE__),
			['jquery'],
			false,
			true
		);

		wp_register_script(
			'accountino-nouislider-script',
			plugins_url('/assets/js/nouislider.min.js', ACCOUNTINO_CORE__FILE__),
			[],
			null,
			true
		);
	}


	/**
	 * Register scripts
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 */
	public function register_styles()
	{
		$suffix = self::get_assets_suffix();

		wp_enqueue_style('widgets-style', ACCOUNTINO_CORE_ASSETS_URL . 'css/widgets' . $suffix . '.css');

		wp_enqueue_style(
			'accountino-nav-menu-style',
			plugins_url('/assets/css/widget-nav-menu' . $suffix . '.css', ACCOUNTINO_CORE__FILE__),
			array()
		);

		wp_enqueue_style(
			'accountino-nouislider-style',
			plugins_url('/assets/css/nouislider.min.css', ACCOUNTINO_CORE__FILE__),
			[],
			null
		);
	}
}

// In tests we run the instance manually.
Plugin::instance();
